/*WIKI* 

Loads an ILL TOF NeXus file into a [[Workspace2D]] with the given name.

This loader calculates the elastic peak position (EPP) on the fly.
In cases where the dispersion peak might be higher than the EPP, it is good practice to load a Vanadium file.

The property FilenameVanadium is optional. If it is present the EPP will be loaded from the Vanadium data.

To date this algorithm only supports: IN4, IN5 and IN6

*WIKI*/
//---------------------------------------------------
// Includes
//---------------------------------------------------
#include "MantidDataHandling/LoadILL.h"
#include "MantidAPI/FileProperty.h"
#include "MantidAPI/Progress.h"
#include "MantidAPI/RegisterFileLoader.h"
#include "MantidGeometry/Instrument.h"
#include "MantidKernel/EmptyValues.h"
#include "MantidKernel/UnitFactory.h"
#include "MantidDataHandling/LoadHelper.h"

#include <limits>
#include <algorithm>
#include <iostream>
#include <vector>
#include <cmath>

namespace Mantid 
{
  namespace DataHandling 
  {

    using namespace Kernel;
    using namespace API;
    using namespace NeXus;

    DECLARE_NEXUS_FILELOADER_ALGORITHM(LoadILL);
    ;

    /**
    * tostring operator to print the contents of NXClassInfo
    *
    * TODO : This has to go somewhere else
    */
    std::ostream& operator<<(std::ostream &strm, const NXClassInfo &c) 
    {
      return strm << "NXClassInfo :: nxname: " << c.nxname << " , nxclass: "
        << c.nxclass;
    }

    /**
    * Return the confidence with with this algorithm can load the file
    * @param descriptor A descriptor for the file
    * @returns An integer specifying the confidence level. 0 indicates it will not be used
    */
    int LoadILL::confidence(Kernel::NexusDescriptor & descriptor) const 
    {

      // fields existent only at the ILL
      if (descriptor.pathExists("/entry0/wavelength")
        && descriptor.pathExists("/entry0/experiment_identifier")
        && descriptor.pathExists("/entry0/mode")
        && !descriptor.pathExists("/entry0/dataSD") // This one is for LoadILLIndirect
        && !descriptor.pathExists("/entry0/instrument/VirtualChopper") // This one is for LoadILLReflectometry
        )
      {
          return 80;
      } 
      else 
      {
        return 0;
      }
    }

    //---------------------------------------------------
    // Private member functions
    //---------------------------------------------------

    LoadILL::LoadILL() :

      API::IFileLoader<Kernel::NexusDescriptor>() 
    {
      m_instrumentName = "";
      m_wavelength = 0;
      m_channelWidth = 0;
      m_numberOfChannels = 0;
      m_numberOfHistograms = 0;
      m_numberOfTubes = 0;
      m_numberOfPixelsPerTube = 0;
      m_monitorElasticPeakPosition = 0;
      m_l1 = 0;
      m_l2 = 0;
      m_supportedInstruments.push_back("IN4");
      m_supportedInstruments.push_back("IN5");
      m_supportedInstruments.push_back("IN6");

    }

    /**
    * Initialise the algorithm
    */
	void LoadILL::init() {
		declareProperty(
				new FileProperty("Filename", "", FileProperty::Load, ".nxs"),
				"File path of the Data file to load");

		declareProperty(
				new FileProperty("FilenameVanadium", "", FileProperty::OptionalLoad,
						".nxs"),
				"File path of the Vanadium file to load (Optional)");

		declareProperty(
				new WorkspaceProperty<API::MatrixWorkspace>("WorkspaceVanadium", "",
						Direction::Input, PropertyMode::Optional),
				"Vanadium Workspace file to load (Optional)");

		declareProperty(
				new WorkspaceProperty<>("OutputWorkspace", "", Direction::Output),
				"The name to use for the output workspace");

	}


    /**
    * Execute the algorithm
    */
    void LoadILL::exec() 
    {
      // Retrieve filename
      std::string filenameData = getPropertyValue("Filename");
      std::string filenameVanadium = getPropertyValue("FilenameVanadium");
      MatrixWorkspace_sptr vanaWS = getProperty("WorkspaceVanadium");

      // open the root node
      NeXus::NXRoot dataRoot(filenameData);
      NXEntry dataFirstEntry = dataRoot.openFirstEntry();

      loadInstrumentDetails(dataFirstEntry);
      loadTimeDetails(dataFirstEntry);
      initWorkSpace(dataFirstEntry);

      runLoadInstrument(); // just to get IDF contents
      initInstrumentSpecific();

      int calculatedDetectorElasticPeakPosition = getEPPFromVanadium(filenameVanadium,vanaWS);

      loadDataIntoTheWorkSpace(dataFirstEntry,calculatedDetectorElasticPeakPosition);

      loadRunDetails(dataFirstEntry);
      loadExperimentDetails(dataFirstEntry);

      // load the instrument from the IDF if it exists
      runLoadInstrument();

      // Set the output workspace property
      setProperty("OutputWorkspace", m_localWorkspace);
    }


    /**
     * Get the elastic peak position (EPP) from a Vanadium Workspace
     * or filename.
     * Returns the EPP
     */
	int LoadILL::getEPPFromVanadium(const std::string &filenameVanadium,
		MatrixWorkspace_sptr vanaWS) {
		int calculatedDetectorElasticPeakPosition = -1;

		if (vanaWS != NULL) {

			// Check if it has been store on the run object for this workspace
			if (vanaWS->run().hasProperty("EPP")) {
				Kernel::Property* prop = vanaWS->run().getProperty("EPP");
				calculatedDetectorElasticPeakPosition = boost::lexical_cast<int>(
						prop->value());
				g_log.information()
						<< "Using EPP from Vanadium WorkSpace : value =  "
						<< calculatedDetectorElasticPeakPosition << "\n";
			} else {
				g_log.error(
						"No EPP Property in the Vanadium Workspace. Following regular procedure...");
				//throw std::invalid_argument("No EPP value has been set or stored within the run information.");
			}
		}
		if (calculatedDetectorElasticPeakPosition == -1 && filenameVanadium != "") {
			g_log.information()
					<< "Calculating the elastic peak position from the Vanadium."
					<< std::endl;
			calculatedDetectorElasticPeakPosition = validateVanadium(
					filenameVanadium);
		}
		return calculatedDetectorElasticPeakPosition;
	}


    /**
    * Set the instrument name along with its path on the nexus file
    */
	void LoadILL::loadInstrumentDetails(NeXus::NXEntry& firstEntry) {

		m_instrumentPath = m_loader.findInstrumentNexusPath(firstEntry);

		if (m_instrumentPath == "") {
			throw std::runtime_error(
					"Cannot set the instrument name from the Nexus file!");
		}

		m_instrumentName = m_loader.getStringFromNexusPath(firstEntry,
				m_instrumentPath + "/name");

		if (std::find(m_supportedInstruments.begin(), m_supportedInstruments.end(),
				m_instrumentName) == m_supportedInstruments.end()) {
			std::string message = "The instrument " + m_instrumentName + " is not valid for this loader!";
			throw std::runtime_error(message);
		}

		g_log.debug() << "Instrument name set to: " + m_instrumentName << std::endl;

	}

    /**
    * Creates the workspace and initialises member variables with
    * the corresponding values
    *
    * @param entry :: The Nexus entry
    *
    */
    void LoadILL::initWorkSpace(NeXus::NXEntry& entry) 
    {

      // read in the data
      NXData dataGroup = entry.openNXData("data");
      NXInt data = dataGroup.openIntData();

      m_numberOfTubes = static_cast<size_t>(data.dim0());
      m_numberOfPixelsPerTube = static_cast<size_t>(data.dim1());
      m_numberOfChannels = static_cast<size_t>(data.dim2());

      // dim0 * m_numberOfPixelsPerTube is the total number of detectors
      m_numberOfHistograms = m_numberOfTubes * m_numberOfPixelsPerTube;

      g_log.debug() << "NumberOfTubes: " << m_numberOfTubes << std::endl;
      g_log.debug() << "NumberOfPixelsPerTube: " << m_numberOfPixelsPerTube
        << std::endl;
      g_log.debug() << "NumberOfChannels: " << m_numberOfChannels << std::endl;

      // Now create the output workspace
      // Might need to get this value from the number of monitors in the Nexus file
      // params:
      // workspace type,
      // total number of spectra + (number of monitors = 0),
      // bin boundaries = m_numberOfChannels + 1
      // Z/time dimension
      m_localWorkspace = WorkspaceFactory::Instance().create("Workspace2D",
        m_numberOfHistograms, m_numberOfChannels + 1, m_numberOfChannels);
      m_localWorkspace->getAxis(0)->unit() = UnitFactory::Instance().create(
        "TOF");
      m_localWorkspace->setYUnitLabel("Counts");

    }

    /**
    * Function to do specific instrument stuff
    *
    */
    void LoadILL::initInstrumentSpecific() 
    {
      m_l1 = m_loader.getL1(m_localWorkspace);
      // this will be mainly for IN5 (flat PSD detector)
      m_l2 = m_loader.getInstrumentProperty(m_localWorkspace,"l2");
      if (m_l2 == EMPTY_DBL()) {
        g_log.debug("Calculating L2 from the IDF.");
        m_l2 = m_loader.getL2(m_localWorkspace);
      }
    }

    /**
    * Load the time details from the nexus file.
    * @param entry :: The Nexus entry
    */
    void LoadILL::loadTimeDetails(NeXus::NXEntry& entry) 
    {

      m_wavelength = entry.getFloat("wavelength");

      // Monitor can be monitor (IN5) or monitor1 (IN6)
      std::string monitorName;
      if (entry.containsGroup("monitor"))
        monitorName = "monitor";
      else if (entry.containsGroup("monitor1"))
        monitorName = "monitor1";
      else 
      {
        std::string message("Cannot find monitor[1] in the Nexus file!");
        g_log.error(message);
        throw std::runtime_error(message);
      }

      m_monitorElasticPeakPosition = entry.getInt(monitorName + "/elasticpeak");

      NXFloat time_of_flight_data = entry.openNXFloat(
        monitorName + "/time_of_flight");
      time_of_flight_data.load();

      // The entry "monitor/time_of_flight", has 3 fields:
      // channel width , number of channels, Time of flight delay
      m_channelWidth = time_of_flight_data[0];
      //	m_timeOfFlightDelay = time_of_flight_data[2];

      g_log.debug("Nexus Data:");
      g_log.debug() << " ChannelWidth: " << m_channelWidth << std::endl;
      g_log.debug() << " Wavelength: " << m_wavelength << std::endl;
      g_log.debug() << " ElasticPeakPosition: " << m_monitorElasticPeakPosition
        << std::endl;

    }



    /**
    * Load information about the run.
    * People from ISIs have this... Let's do the same for the ILL.
    * TODO: They also have a lot of info in XML format!
    *
    * @param entry :: The Nexus entry
    */
    void LoadILL::loadRunDetails(NXEntry & entry) 
    {

      API::Run & runDetails = m_localWorkspace->mutableRun();

      int runNum = entry.getInt("run_number");
      std::string run_num = boost::lexical_cast<std::string>(runNum);
      runDetails.addProperty("run_number", run_num);

      std::string start_time = entry.getString("start_time");
      start_time = m_loader.dateTimeInIsoFormat(start_time);
      runDetails.addProperty("run_start", start_time);

      std::string end_time = entry.getString("end_time");
      end_time = m_loader.dateTimeInIsoFormat(end_time);
      runDetails.addProperty("run_end", end_time);

      //m_wavelength = entry.getFloat("wavelength");
      std::string wavelength = boost::lexical_cast<std::string>(m_wavelength);
      //runDetails.addProperty<double>("wavelength", m_wavelength);
      runDetails.addProperty("wavelength", wavelength);
      double ei = m_loader.calculateEnergy(m_wavelength);
      runDetails.addProperty<double>("Ei", ei, true); //overwrite
      //std::string ei_str = boost::lexical_cast<std::string>(ei);
      //runDetails.addProperty("Ei", ei_str);

      std::string duration = boost::lexical_cast<std::string>(
        entry.getFloat("duration"));
      runDetails.addProperty("duration", duration);

      std::string preset = boost::lexical_cast<std::string>(
        entry.getFloat("preset"));
      runDetails.addProperty("preset", preset);

      std::string mode = entry.getString("mode");
      runDetails.addProperty("mode", mode);

      std::string title = entry.getString("title");
      runDetails.addProperty("title", title);
      m_localWorkspace->setTitle(title);

      // Below : This should belong to sample ???
      std::string experiment_identifier = entry.getString(
        "experiment_identifier");
      runDetails.addProperty("experiment_identifier", experiment_identifier);
      m_localWorkspace->mutableSample().setName(experiment_identifier);

      std::string temperature = boost::lexical_cast<std::string>(
        entry.getFloat("sample/temperature"));
      runDetails.addProperty("temperature", temperature);
    }

    /*
    * Load data about the Experiment.
    *
    * TODO: This is very incomplete. In ISIS they much more info in the nexus file than ILL.
    *
    * @param entry :: The Nexus entry
    */
    void LoadILL::loadExperimentDetails(NXEntry & entry) 
    {

      // TODO: Do the rest
      // Pick out the geometry information

      std::string description = boost::lexical_cast<std::string>(
        entry.getFloat("sample/description"));

      m_localWorkspace->mutableSample().setName(description);

      //	m_localWorkspace->mutableSample().setThickness(static_cast<double> (isis_raw->spb.e_thick));
      //	m_localWorkspace->mutableSample().setHeight(static_cast<double> (isis_raw->spb.e_height));
      //	m_localWorkspace->mutableSample().setWidth(static_cast<double> (isis_raw->spb.e_width));

    }

    /**
    * Gets the experimental Elastic Peak Position in the dectector
    * as the value parsed from the nexus file might be wrong.
    *
    * It gets a few spectra in the equatorial line of the detector,
    * sum them up and finds the maximum = the Elastic peak
    *
    *
    * @param data :: spectra data
    * @return detector Elastic Peak Position
    */
    int LoadILL::getDetectorElasticPeakPosition(const NeXus::NXInt &data) 
    {

      // j = index in the equatorial line (256/2=128)
      // both index 127 and 128 are in the equatorial line
      size_t j = m_numberOfPixelsPerTube / 2;

      // ignore the first tubes and the last ones to avoid the beamstop
      //get limits in the m_numberOfTubes
      size_t tubesToRemove = m_numberOfTubes / 7;

      std::vector<int> cumulatedSumOfSpectras(m_numberOfChannels, 0);
      for (size_t i = tubesToRemove; i < m_numberOfTubes - tubesToRemove; i++) {
        int* data_p = &data(static_cast<int>(i), static_cast<int>(j), 0);
        std::vector<int> thisSpectrum(data_p, data_p + m_numberOfChannels);
        // sum spectras
        std::transform(thisSpectrum.begin(), thisSpectrum.end(),
          cumulatedSumOfSpectras.begin(), cumulatedSumOfSpectras.begin(),
          std::plus<int>());
      }
      auto it = std::max_element(cumulatedSumOfSpectras.begin(),
        cumulatedSumOfSpectras.end());

      int calculatedDetectorElasticPeakPosition;
      if (it == cumulatedSumOfSpectras.end()) 
      {
        g_log.warning()
          << "No Elastic peak position found! Assuming the EPP in the Nexus file: "
          << m_monitorElasticPeakPosition << std::endl;
        calculatedDetectorElasticPeakPosition = m_monitorElasticPeakPosition;

      } 
      else 
      {
        //calculatedDetectorElasticPeakPosition = *it;
        calculatedDetectorElasticPeakPosition = static_cast<int>(std::distance(
          cumulatedSumOfSpectras.begin(), it));

        if (calculatedDetectorElasticPeakPosition == 0) 
        {
          g_log.warning()
            << "Elastic peak position is ZERO Assuming the EPP in the Nexus file: "
            << m_monitorElasticPeakPosition << std::endl;
          calculatedDetectorElasticPeakPosition =
            m_monitorElasticPeakPosition;

        } 
        else 
        {
          g_log.debug() << "Calculated Detector EPP: "
            << calculatedDetectorElasticPeakPosition;
          g_log.debug() << " :: Read EPP from the nexus file: "
            << m_monitorElasticPeakPosition << std::endl;
        }
      }
      return calculatedDetectorElasticPeakPosition;

    }

	/**
	 * It loads the vanadium nexus file and cross checks it against the
	 * data file already loaded (same wavelength and same instrument configuration).
	 * If matches looks for the elastic peak in the vanadium file and returns
	 * it position.
	 *
	 * @param filenameVanadium :: The path for the vanadium nexus file.
	 * @return The elastic peak position inside the tof channels.
	 */
	int LoadILL::validateVanadium(const std::string &filenameVanadium) {
		NeXus::NXRoot vanaRoot(filenameVanadium);
		NXEntry vanaFirstEntry = vanaRoot.openFirstEntry();

		double wavelength = vanaFirstEntry.getFloat("wavelength");

		// read in the data
		NXData dataGroup = vanaFirstEntry.openNXData("data");
		NXInt data = dataGroup.openIntData();

		size_t numberOfTubes = static_cast<size_t>(data.dim0());
		size_t numberOfPixelsPerTube = static_cast<size_t>(data.dim1());
		size_t numberOfChannels = static_cast<size_t>(data.dim2());

		if (wavelength != m_wavelength || numberOfTubes != m_numberOfTubes
				|| numberOfPixelsPerTube != m_numberOfPixelsPerTube
				|| numberOfChannels != m_numberOfChannels) {
			throw std::runtime_error(
					"Vanadium and Data were not collected in the same conditions!");
		}

		data.load();
		int calculatedDetectorElasticPeakPosition = getDetectorElasticPeakPosition(
				data);
		return calculatedDetectorElasticPeakPosition;
	}

    /**
    * Loads all the spectra into the workspace, including that from the monitor
    *
    * @param entry :: The Nexus entry
    * @param vanaCalculatedDetectorElasticPeakPosition :: If -1 uses this value as the elastic peak position at the detector.
    */
    void LoadILL::loadDataIntoTheWorkSpace(NeXus::NXEntry& entry, int vanaCalculatedDetectorElasticPeakPosition)
    {

      // read in the data
      NXData dataGroup = entry.openNXData("data");
      NXInt data = dataGroup.openIntData();
      // load the counts from the file into memory
      data.load();
      /*
      * Detector: Find real elastic peak in the detector.
      * Looks for a few elastic peaks on the equatorial line of the detector.
      */
      int calculatedDetectorElasticPeakPosition;
      if (vanaCalculatedDetectorElasticPeakPosition == -1)
    	  calculatedDetectorElasticPeakPosition = getDetectorElasticPeakPosition(data);
      else
    	  calculatedDetectorElasticPeakPosition = vanaCalculatedDetectorElasticPeakPosition;

      //set it as a Property
      API::Run & runDetails = m_localWorkspace->mutableRun();
      runDetails.addProperty("EPP", calculatedDetectorElasticPeakPosition);

      double theoreticalElasticTOF = (m_loader.calculateTOF(m_l1,m_wavelength) + m_loader.calculateTOF(m_l2,m_wavelength))
        * 1e6; //microsecs

      // Calculate the real tof (t1+t2) put it in tof array
      std::vector<double> detectorTofBins(m_numberOfChannels + 1);
      for (size_t i = 0; i < m_numberOfChannels + 1; ++i) 
      {
        detectorTofBins[i] = theoreticalElasticTOF
          + m_channelWidth
          * static_cast<double>(static_cast<int>(i)
          - calculatedDetectorElasticPeakPosition)
          - m_channelWidth / 2; // to make sure the bin is in the middle of the elastic peak

      }
      //g_log.debug() << "Detector TOF bins: ";
      //for (auto i : detectorTofBins) g_log.debug() << i << " ";
      //g_log.debug() << "\n";

      g_log.information() << "T1+T2 : Theoretical = " << theoreticalElasticTOF;
      g_log.information() << " ::  Calculated bin = ["
        << detectorTofBins[calculatedDetectorElasticPeakPosition] << ","
        << detectorTofBins[calculatedDetectorElasticPeakPosition + 1] << "]"
        << std::endl;

      // Assign calculated bins to first X axis
      m_localWorkspace->dataX(0).assign(detectorTofBins.begin(),
        detectorTofBins.end());

      Progress progress(this, 0, 1, m_numberOfTubes * m_numberOfPixelsPerTube);
      size_t spec = 0;
      for (size_t i = 0; i < m_numberOfTubes; ++i) 
      {
        for (size_t j = 0; j < m_numberOfPixelsPerTube; ++j) 
        {
          if (spec > 0) 
          {
            // just copy the time binning axis to every spectra
            m_localWorkspace->dataX(spec) = m_localWorkspace->readX(0);
          }
          // Assign Y
          int* data_p = &data(static_cast<int>(i), static_cast<int>(j), 0);
          m_localWorkspace->dataY(spec).assign(data_p,
            data_p + m_numberOfChannels);

          // Assign Error
          MantidVec& E = m_localWorkspace->dataE(spec);
          std::transform(data_p, data_p + m_numberOfChannels, E.begin(),
            LoadILL::calculateError);

          ++spec;
          progress.report();
        }
      }
    }

    /**
    * Run the Child Algorithm LoadInstrument.
    */
	void LoadILL::runLoadInstrument() {

		IAlgorithm_sptr loadInst = createChildAlgorithm("LoadInstrument");

		// Now execute the Child Algorithm. Catch and log any error, but don't stop.
		try {

			// TODO: depending on the m_numberOfPixelsPerTube we might need to load a different IDF

			loadInst->setPropertyValue("InstrumentName", m_instrumentName);
			loadInst->setProperty<MatrixWorkspace_sptr>("Workspace",
					m_localWorkspace);
			loadInst->execute();
		} catch (...) {
			g_log.information("Cannot load the instrument definition.");
		}
	}



  } // namespace DataHandling
} // namespace Mantid
